<?php
session_start();
header('Content-Type: application/json');

// Verificar si el usuario está logueado
if (!isset($_SESSION['usuario_logueado']) || $_SESSION['usuario_logueado'] !== true) {
    echo json_encode(['success' => false, 'message' => 'Usuario no autenticado']);
    exit();
}

// Incluir conexión a base de datos
require_once 'conexion.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) {
        throw new Exception('Datos de entrada inválidos');
    }

    $plaza_id = $input['plaza_id'] ?? null;
    $fecha = $input['fecha'] ?? null;
    $hora = $input['hora'] ?? null;
    $duracion = $input['duracion'] ?? 8;
    $usuario_id = $_SESSION['usuario_id'];

    if (!$plaza_id || !$fecha || !$hora) {
        throw new Exception('Faltan datos obligatorios');
    }

    // Validar que la plaza existe y es reservable
    $stmt = $conexion->prepare("SELECT id, numero, tipo, reservable FROM plazas WHERE id = ?");
    $stmt->bind_param("i", $plaza_id);
    $stmt->execute();
    $resultado = $stmt->get_result();
    $plaza = $resultado->fetch_assoc();
    $stmt->close();

    if (!$plaza) {
        throw new Exception('La plaza no existe');
    }

    if (!$plaza['reservable']) {
        throw new Exception('Esta plaza no es reservable');
    }

    $fecha_reserva = ($fecha === 'hoy') ? date('Y-m-d') : date('Y-m-d', strtotime('+1 day'));

    if ($fecha === 'hoy' && $hora < date('H:i')) {
        throw new Exception('No se pueden hacer reservas para horas pasadas');
    }

    $hora_inicio = $hora . ':00';
    $hora_fin = date('H:i:s', strtotime($hora_inicio . ' +' . $duracion . ' hours'));

    // Verificar conflictos con otras reservas
    $stmt = $conexion->prepare("
        SELECT COUNT(*) as conflictos
        FROM reservas 
        WHERE plaza_id = ? 
        AND fecha_reserva = ? 
        AND estado = 'activa'
        AND (
            (hora_inicio <= ? AND hora_fin > ?) OR
            (hora_inicio < ? AND hora_fin >= ?) OR
            (hora_inicio >= ? AND hora_inicio < ?)
        )
    ");
    $stmt->bind_param("isssssss", $plaza_id, $fecha_reserva,
        $hora_inicio, $hora_inicio,
        $hora_fin, $hora_fin,
        $hora_inicio, $hora_fin
    );
    $stmt->execute();
    $resultado = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if ($resultado['conflictos'] > 0) {
        throw new Exception('La plaza ya está reservada para esa fecha y hora');
    }

    // Verificar si el usuario ya tiene una reserva para esa hora
    $stmt = $conexion->prepare("
        SELECT COUNT(*) as mis_reservas
        FROM reservas 
        WHERE usuario_id = ? 
        AND fecha_reserva = ? 
        AND estado = 'activa'
        AND (
            (hora_inicio <= ? AND hora_fin > ?) OR
            (hora_inicio < ? AND hora_fin >= ?) OR
            (hora_inicio >= ? AND hora_inicio < ?)
        )
    ");
    $stmt->bind_param("isssssss", $usuario_id, $fecha_reserva,
        $hora_inicio, $hora_inicio,
        $hora_fin, $hora_fin,
        $hora_inicio, $hora_fin
    );
    $stmt->execute();
    $resultado = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if ($resultado['mis_reservas'] > 0) {
        throw new Exception('Ya tienes una reserva para esa fecha y hora');
    }

    // Límite de reservas por día
    $stmt = $conexion->prepare("
        SELECT COUNT(*) as reservas_dia
        FROM reservas 
        WHERE usuario_id = ? 
        AND fecha_reserva = ? 
        AND estado = 'activa'
    ");
    $stmt->bind_param("is", $usuario_id, $fecha_reserva);
    $stmt->execute();
    $resultado = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if ($resultado['reservas_dia'] >= 2) {
        throw new Exception('Máximo 2 reservas por día permitidas');
    }

    // Iniciar transacción
    $conexion->begin_transaction();

    // Insertar reserva
    $stmt = $conexion->prepare("
        INSERT INTO reservas (usuario_id, plaza_id, fecha_reserva, hora_inicio, hora_fin, estado)
        VALUES (?, ?, ?, ?, ?, 'activa')
    ");
    $stmt->bind_param("iisss", $usuario_id, $plaza_id, $fecha_reserva, $hora_inicio, $hora_fin);
    $stmt->execute();
    $reserva_id = $stmt->insert_id;
    $stmt->close();

    // Confirmar transacción
    $conexion->commit();

    // Obtener datos de la reserva
    $stmt = $conexion->prepare("
        SELECT r.*, p.numero as plaza_numero, p.tipo as plaza_tipo, u.nombre as usuario_nombre
        FROM reservas r
        JOIN plazas p ON r.plaza_id = p.id
        JOIN usuarios u ON r.usuario_id = u.id
        WHERE r.id = ?
    ");
    $stmt->bind_param("i", $reserva_id);
    $stmt->execute();
    $reserva = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    echo json_encode([
        'success' => true,
        'message' => 'Reserva creada exitosamente',
        'reserva' => [
            'id' => $reserva['id'],
            'plaza_numero' => $reserva['plaza_numero'],
            'plaza_tipo' => $reserva['plaza_tipo'],
            'fecha_reserva' => $reserva['fecha_reserva'],
            'hora_inicio' => $reserva['hora_inicio'],
            'hora_fin' => $reserva['hora_fin']
        ]
    ]);

} catch (Exception $e) {
    if ($conexion->errno) {
        $conexion->rollback();
    }
    error_log("Error en reservar.php: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
