// Variables globales
let fechaSeleccionada = 'hoy';
let horaSeleccionada = '08:00';
let plazaSeleccionada = null;

// Elementos del DOM
const modal = document.getElementById('modal-reserva');
const modalClose = document.getElementsByClassName('close')[0];
const formReserva = document.getElementById('form-reserva');
const btnActualizar = document.getElementById('actualizar');
const selectFecha = document.getElementById('fecha');
const selectHora = document.getElementById('hora');
const loader = document.getElementById('loader-container');

// Inicialización
document.addEventListener('DOMContentLoaded', function() {
    inicializarEventos();
    actualizarMapa();
    cargarMisReservas();
});

// Event listeners
function inicializarEventos() {
    // Filtros
    selectFecha.addEventListener('change', function() {
        fechaSeleccionada = this.value;
        actualizarMapa();
    });
    
    selectHora.addEventListener('change', function() {
        horaSeleccionada = this.value;
        actualizarMapa();
    });
    
    
    // Modal
    modalClose.addEventListener('click', cerrarModal);
    
    window.addEventListener('click', function(event) {
        if (event.target === modal) {
            cerrarModal();
        }
    });
    
    // Formulario de reserva
    formReserva.addEventListener('submit', confirmarReserva);
    
    // Botón cancelar del modal
    document.querySelector('.btn-cancelar').addEventListener('click', cerrarModal);
    
    // Plazas clickeables
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('clickable') && 
            e.target.classList.contains('disponible')) {
            abrirModalReserva(e.target);
        }
    });
    
    // Actualizar cada 60 segundos
    setInterval(actualizarMapa, 60000);
}

// Actualizar el mapa con disponibilidad
async function actualizarMapa() {
    try {
        mostrarLoader(true);
        
        const response = await fetch('consultar_disponibilidad.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                fecha: fechaSeleccionada,
                hora: horaSeleccionada
            })
        });
        
        if (!response.ok) {
            throw new Error('Error en la respuesta del servidor');
        }
        
        const data = await response.json();
        
        if (data.success) {
            actualizarEstadoPlazas(data.plazas, data.misReservas);
        } else {
            mostrarError('Error al consultar disponibilidad: ' + data.message);
        }
        
    } catch (error) {
        console.error('Error:', error);
        mostrarError('Error de conexión. Inténtalo de nuevo.');
    } finally {
        mostrarLoader(false);
    }
}

// Actualizar estado visual de las plazas
function actualizarEstadoPlazas(plazas, misReservas) {
    const plazasElements = document.querySelectorAll('.plaza[data-id]');
    
    plazasElements.forEach(plazaEl => {
        const plazaId = parseInt(plazaEl.dataset.id);
        const plaza = plazas.find(p => p.id === plazaId);
        
        if (plaza) {
            // Remover clases de estado previas
            plazaEl.classList.remove('disponible', 'ocupada', 'mi-reserva');
            
            // Verificar si es mi reserva
            const esMiReserva = misReservas.some(r => r.plaza_id === plazaId);
            
            if (esMiReserva) {
                plazaEl.classList.add('ocupada', 'mi-reserva');
            } else if (plaza.ocupada) {
                plazaEl.classList.add('ocupada');
            } else {
                plazaEl.classList.add('disponible');
            }
        }
    });
}

// Abrir modal de reserva
function abrirModalReserva(plazaElement) {
    plazaSeleccionada = {
        id: parseInt(plazaElement.dataset.id),
        numero: plazaElement.dataset.numero,
        tipo: plazaElement.dataset.tipo
    };
    
    // Llenar información del modal
    document.getElementById('modal-plaza').textContent = plazaSeleccionada.numero;
    document.getElementById('modal-fecha').textContent = 
        fechaSeleccionada === 'hoy' ? 'Hoy' : 'Mañana';
    document.getElementById('modal-hora').textContent = horaSeleccionada;
    
    modal.style.display = 'block';
}

// Cerrar modal
function cerrarModal() {
    modal.style.display = 'none';
    plazaSeleccionada = null;
}

// Confirmar reserva
async function confirmarReserva(e) {
    e.preventDefault();
    
    if (!plazaSeleccionada) {
        mostrarError('No hay plaza seleccionada');
        return;
    }
    
    const duracion = parseInt(document.getElementById('duracion').value);
    
    try {
        mostrarLoader(true);
        
        const response = await fetch('reservar.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                plaza_id: plazaSeleccionada.id,
                fecha: fechaSeleccionada,
                hora: horaSeleccionada,
                duracion: duracion
            })
        });
        
        if (!response.ok) {
            throw new Error('Error en la respuesta del servidor');
        }
        
        const data = await response.json();
        
        if (data.success) {
            mostrarExito('¡Reserva confirmada exitosamente!');
            
            // Guardar id antes de cerrar modal
            const plazaId = plazaSeleccionada.id;
            
            cerrarModal();
            actualizarMapa();
            cargarMisReservas();
            
            // Añadir clase de éxito para feedback visual
            const plazaEl = document.querySelector(`[data-id="${plazaId}"]`);
            if (plazaEl) {
                plazaEl.classList.add('success');
                setTimeout(() => {
                    plazaEl.classList.remove('success');
                }, 1000);
            }
        } else {
            mostrarError('Error al reservar: ' + data.message);
        }
        
    } catch (error) {
        console.error('Error:', error);
        mostrarError('Error de conexión. Inténtalo de nuevo.');
    } finally {
        mostrarLoader(false);
    }
}

// Cargar mis reservas
async function cargarMisReservas() {
    try {
        const response = await fetch('mis_reservas.php');
        
        if (!response.ok) {
            throw new Error('Error en la respuesta del servidor');
        }
        
        const data = await response.json();
        
        if (data.success) {
            mostrarMisReservas(data.reservas);
        } else {
            document.getElementById('lista-reservas').innerHTML = 
                '<p>Error al cargar reservas: ' + data.message + '</p>';
        }
        
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('lista-reservas').innerHTML = 
            '<p>Error de conexión al cargar reservas.</p>';
    }
}

// Mostrar mis reservas en el DOM
function mostrarMisReservas(reservas) {
    const container = document.getElementById('lista-reservas');
    
    if (reservas.length === 0) {
        container.innerHTML = '<p>No tienes reservas activas.</p>';
        return;
    }
    
    let html = '';
    reservas.forEach(reserva => {
        const fechaFormateada = formatearFecha(reserva.fecha_reserva);
        const horaInicio = reserva.hora_inicio.substring(0, 5);
        const horaFin = reserva.hora_fin.substring(0, 5);
        
        html += `
            <div class="reserva-item">
                <div class="reserva-info">
                    <strong>Plaza ${reserva.plaza_numero}</strong> 
                    (${reserva.plaza_tipo.toUpperCase()}) <br>
                    <small>${fechaFormateada} de ${horaInicio} a ${horaFin}</small>
                </div>
                <div class="reserva-acciones">
                    <button class="btn-cancelar-reserva" 
                            onclick="cancelarReserva(${reserva.id})">
                        🗑️ Cancelar
                    </button>
                </div>
            </div>
        `;
    });
    
    container.innerHTML = html;
}

// Cancelar reserva
async function cancelarReserva(reservaId) {
    if (!confirm('¿Estás seguro de que quieres cancelar esta reserva?')) {
        return;
    }
    
    try {
        mostrarLoader(true);
        
        const response = await fetch('cancelar_reserva.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                reserva_id: reservaId
            })
        });
        
        if (!response.ok) {
            throw new Error('Error en la respuesta del servidor');
        }
        
        const data = await response.json();
        
        if (data.success) {
            mostrarExito('Reserva cancelada exitosamente');
            cargarMisReservas();
            actualizarMapa();
        } else {
            mostrarError('Error al cancelar reserva: ' + data.message);
        }
        
    } catch (error) {
        console.error('Error:', error);
        mostrarError('Error de conexión. Inténtalo de nuevo.');
    } finally {
        mostrarLoader(false);
    }
}

// Utilidades
/*function mostrarLoading(mostrar) {
    loading.style.display = mostrar ? 'flex' : 'none';
}*/
function mostrarLoader(mostrar) {
    loader.style.display = mostrar ? 'flex' : 'none';
}
function mostrarError(mensaje) {
    alert('❌ ' + mensaje);
}

function mostrarExito(mensaje) {
    alert('✅ ' + mensaje);
}

function formatearFecha(fecha) {
    const date = new Date(fecha + 'T00:00:00');
    const hoy = new Date();
    const manana = new Date();
    manana.setDate(hoy.getDate() + 1);
    
    if (date.toDateString() === hoy.toDateString()) {
        return 'Hoy';
    } else if (date.toDateString() === manana.toDateString()) {
        return 'Mañana';
    } else {
        return date.toLocaleDateString('es-ES', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    }
}

// Función para hacer las plazas accesibles por teclado
document.addEventListener('keydown', function(e) {
    const focusedElement = document.activeElement;
    
    if (focusedElement.classList.contains('clickable') && 
        focusedElement.classList.contains('disponible') && 
        (e.key === 'Enter' || e.key === ' ')) {
        e.preventDefault();
        abrirModalReserva(focusedElement);
    }
});

// Hacer las plazas focusables
document.addEventListener('DOMContentLoaded', function() {
    const plazasClickeables = document.querySelectorAll('.clickable');
    plazasClickeables.forEach(plaza => {
        plaza.setAttribute('tabindex', '0');
        plaza.setAttribute('role', 'button');
        plaza.setAttribute('aria-label', 
            `Plaza ${plaza.dataset.numero}, ${plaza.dataset.tipo}, disponible para reservar`);
    });
});